/***************************************************************************
 * This file is part of the KDE project
 * copyright (C) 2005 by Sebastian Sauer (mail@dipe.org)
 * copyright (C) 2005 by Tobi Krebs (tobi.krebs@gmail.com)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "metaobject.h"
#include "metamethod.h"
#include "variable.h"
#include "exception.h"

#include <tqguardedptr.h>
#include <tqmetaobject.h>

#include <kdebug.h>

using namespace KoMacro;

namespace KoMacro {

	/**
	* @internal d-pointer class to be more flexible on future extension of the
	* functionality without to much risk to break the binary compatibility.
	*/
	class MetaObject::Private
	{
		public:

			/**
			* The TQObject instance this @a MetaObject belongs to.
			*/
			TQGuardedPtr<TQObject> const object;

			/**
			* Constructor.
			*/
			Private(TQObject* const object)
				: object(object)
			{
			}
	};

}

MetaObject::MetaObject(TQObject* const object)
	: TDEShared()
	, d( new Private(object) ) // create the private d-pointer instance.
{
}

MetaObject::~MetaObject()
{
	delete d;
}

TQObject* const MetaObject::object() const
{
	if(! d->object) {
		throw Exception(TQString("Object is undefined."));
	}
	return d->object;
}

/*
TQStrList MetaObject::signalNames() const
{
	return object()->metaObject()->signalNames();
}

TQStrList MetaObject::slotNames() const
{
	return object()->metaObject()->slotNames();
}
*/

int MetaObject::indexOfSignal(const char* signal) const
{
	TQMetaObject* metaobject = object()->metaObject();
	int signalid = metaobject->findSignal(signal, false);
	if(signalid < 0) {
		throw Exception(TQString("Invalid signal \"%1\"").arg(signal));
	}
	return signalid;
}

int MetaObject::indexOfSlot(const char* slot) const
{
	TQMetaObject* metaobject = object()->metaObject();
	int slotid = metaobject->findSlot(slot, false);
	if(slotid < 0) {
		throw Exception(TQString("Invalid slot \"%1\"").arg(slot));
	}
	return slotid;
}

TDESharedPtr<MetaMethod> MetaObject::method(int index)
{
	TQObject* obj = object();
	MetaMethod::Type type = MetaMethod::Slot;
	TQMetaObject* metaobject = obj->metaObject();

	const TQMetaData* metadata = metaobject->slot(index, true);
	if(! metadata) {
		// Try to get a signal with that index iff we failed to determinate
		// a matching slot.

		metadata = metaobject->signal(index, true);
		if(! metadata) {
			throw Exception(TQString("Invalid method index \"%1\" in object \"%2\"").arg(index).arg(obj->name()));
		}
		type = MetaMethod::Signal;
	}

	if(metadata->access != TQMetaData::Public) {
		throw Exception(TQString("Not allowed to access method \"%1\" in object \"%2\"").arg(metadata->name).arg(obj->name()));
	}

	return new MetaMethod(metadata->name, type, this);
}

TDESharedPtr<MetaMethod> MetaObject::signal(const char* signal)
{
	return method( indexOfSignal(signal) );
}

TDESharedPtr<MetaMethod> MetaObject::slot(const char* slot)
{
	return method( indexOfSlot(slot) );
}

TDESharedPtr<Variable> MetaObject::invokeMethod(int index, Variable::List arguments)
{
	// kdDebug() << "MetaObject::invokeMethod(int index, Variable::List arguments)" << endl;
	TDESharedPtr<MetaMethod> m = method(index);
	// kdDebug() << "MetaObject::invokeMethod(int index, Variable::List arguments) return" << endl;
	return m->invoke(arguments);
}

