/***************************************************************************
 * This file is part of the KDE project
 * copyright (C) 2005 by Sebastian Sauer (mail@dipe.org)
 * copyright (C) 2005 by Tobi Krebs (tobi.krebs@gmail.com)
 * copyright (C) 2006 by Sascha Kupper (kusato@kfnv.de)
 *
 * This program is free software; you can redistribute it and/or
 * modify it under the terms of the GNU Library General Public
 * License as published by the Free Software Foundation; either
 * version 2 of the License, or (at your option) any later version.
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
 * Library General Public License for more details.
 * You should have received a copy of the GNU Library General Public License
 * along with this program; see the file COPYING.  If not, write to
 * the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
 ***************************************************************************/

#include "action.h"

#include <kdebug.h>

using namespace KoMacro;

namespace KoMacro {

	/**
	* @internal d-pointer class to be more flexible on future extension of the
	* functionality without to much risk to break the binary compatibility.
	*/
	class Action::Private
	{
		public:

			/**
			* The name this @a Action has.
			*/
			TQString name;

			/**
			* The i18n-caption text this @a Action has.
			*/
			TQString text;

			/**
			* The comment the user is able to define for each action.
			*/
			TQString comment;

			/**
			 * A map of @a Variable instances this @a Action
			 * provides accessible by there TQString name.
			 */
			Variable::Map varmap;

			/**
			* List of variablenames. This list provides a
			* sorted order for the @a Variable instances
			* defined in the map above.
			*/
			TQStringList varnames;

	};

}

Action::Action(const TQString& name, const TQString& text)
	: TQObject()
	, TDEShared()
	, d( new Private() ) // create the private d-pointer instance.
{
	kdDebug() << "Action::Action() name=" << name << endl;
	d->name = name;
	setText(text);

	// Publish this action.
	KoMacro::Manager::self()->publishAction( TDESharedPtr<Action>(this) );
}

Action::~Action()
{
	//kdDebug() << TQString("Action::~Action() name=\"%1\"").arg(name()) << endl;

	// destroy the private d-pointer instance.
	delete d;
}

const TQString Action::toString() const
{
	return TQString("Action:%1").arg(name());
}

const TQString Action::name() const
{
	return d->name;
}

void Action::setName(const TQString& name)
{
	d->name = name;
}

const TQString Action::text() const
{
	return d->text;
}

void Action::setText(const TQString& text)
{
	d->text = text;
}

const TQString Action::comment() const
{
	return d->comment;
}

void Action::setComment(const TQString& comment)
{
	d->comment = comment;
}

bool Action::hasVariable(const TQString& name) const
{
	return d->varmap.contains(name);
}

TDESharedPtr<Variable> Action::variable(const TQString& name) const
{
	return d->varmap.contains(name) ? d->varmap[name] : TDESharedPtr<Variable>(0);
}

Variable::Map Action::variables() const
{
	return d->varmap;
}

TQStringList Action::variableNames() const
{
	return d->varnames;
}

void Action::setVariable(TDESharedPtr<Variable> variable)
{
	const TQString name = variable->name();
	if(! d->varmap.contains(name)) {
		d->varnames.append(name);
	}
	d->varmap.replace(name, variable);
}

void Action::setVariable(const TQString& name, const TQString& text, const TQVariant& variant)
{
	Variable* variable = new Variable(variant);
	variable->setName(name);
	variable->setText(text);
	setVariable( TDESharedPtr<Variable>(variable) );
}

void Action::removeVariable(const TQString& name)
{
	if(d->varmap.contains(name)) {
		d->varmap.remove(name);
		d->varnames.remove(name);
	}
}

#include "action.moc"
