/* This file is part of the KDE project
   Copyright (C) 2004 Cedric Pasteur <cedric.pasteur@free.fr>

   This library is free software; you can redistribute it and/or
   modify it under the terms of the GNU Library General Public
   License as published by the Free Software Foundation; either
   version 2 of the License, or (at your option) any later version.

   This library is distributed in the hope that it will be useful,
   but WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   Library General Public License for more details.

   You should have received a copy of the GNU Library General Public License
   along with this library; see the file COPYING.LIB.  If not, write to
   the Free Software Foundation, Inc., 51 Franklin Street, Fifth Floor,
 * Boston, MA 02110-1301, USA.
*/

#include <tqdom.h>
#include <kdebug.h>

#include "events.h"

namespace KFormDesigner {

Connection::Connection(const TQString &sender, const TQString &signal,
	 const TQString &receiver, const TQString &slot)
{
	m_sender = sender;
	m_signal = signal;
	m_receiver = receiver;
	m_slot = slot;
}

 ///////////////////////////////////////

ConnectionBuffer::ConnectionBuffer()
{
	setAutoDelete(true);
}

void
ConnectionBuffer::fixName(const TQString &oldName, const TQString &newName)
{
	for(Connection *c = first(); c; c = next())
	{
		if(c->sender() == oldName)
			c->setSender(newName);
		if(c->receiver() == oldName)
			c->setReceiver(newName);
	}
}

ConnectionBuffer*
ConnectionBuffer::allConnectionsForWidget(const TQString &widget)
{
	ConnectionBuffer *list = new ConnectionBuffer();
	list->setAutoDelete(false); // or it will delete all our connections
	for(Connection *c = first(); c; c = next())
	{
		if((c->sender() == widget) || (c->receiver() == widget))
			list->append(c);
	}

	return list;
}

void
ConnectionBuffer::save(TQDomNode &parentNode)
{
	if(isEmpty())
		return;

	TQDomDocument domDoc = parentNode.ownerDocument();
	TQDomElement connections;
	if(!parentNode.namedItem("connections").isNull())
		connections = parentNode.namedItem("connections").toElement();
	else
		connections = domDoc.createElement("connections");
	parentNode.appendChild(connections);

	for(Connection *c = first(); c; c = next())
	{
		TQDomElement connection = domDoc.createElement("connection");
		connection.setAttribute("language", "C++");
		connections.appendChild(connection);

		TQDomElement sender = domDoc.createElement("sender");
		connection.appendChild(sender);
		TQDomText senderText = domDoc.createTextNode(c->sender());
		sender.appendChild(senderText);

		TQDomElement signal = domDoc.createElement("signal");
		connection.appendChild(signal);
		TQDomText signalText = domDoc.createTextNode(c->signal());
		signal.appendChild(signalText);

		TQDomElement receiver = domDoc.createElement("receiver");
		connection.appendChild(receiver);
		TQDomText receiverText = domDoc.createTextNode(c->receiver());
		receiver.appendChild(receiverText);

		TQDomElement slot = domDoc.createElement("slot");
		connection.appendChild(slot);
		TQDomText slotText = domDoc.createTextNode(c->slot());
		slot.appendChild(slotText);
	}
}

void
ConnectionBuffer::saveAllConnectionsForWidget(const TQString &widget, TQDomNode parentNode)
{
	ConnectionBuffer *buff = allConnectionsForWidget(widget);
	buff->save(parentNode);
	delete buff;
}

void
ConnectionBuffer::load(TQDomNode node)
{
	for(TQDomNode n = node.firstChild(); !n.isNull(); n = n.nextSibling())
	{
		Connection *conn = new Connection();
		conn->setSender(n.namedItem("sender").toElement().text());
		conn->setSignal(n.namedItem("signal").toElement().text());
		conn->setReceiver(n.namedItem("receiver").toElement().text());
		conn->setSlot(n.namedItem("slot").toElement().text());
		append(conn);
	}
}

void
ConnectionBuffer::removeAllConnectionsForWidget(const TQString &widget)
{
	for(Connection *c = first(); c; c = next())
	{
		if((c->sender() == widget) || (c->receiver() == widget))
			removeRef(c);
	}
}

}

//#include "events.moc"
