// (C) 2005 Max Howell (max.howell@methylblue.com)
// See COPYING file for licensing information

#include <kiconloader.h>
#include <tdelocale.h>
#include <tdetoolbar.h>
#include <tqevent.h>
#include <tqlabel.h>
#include <tqlayout.h>
#include <tqslider.h>
#include <tqtoolbutton.h>
#include <tqtooltip.h>

#include "../debug.h"
#include "volumeAction.h"

#include "xineEngine.h"


class VolumeSlider : public TQFrame
{
public:
   VolumeSlider( TQWidget *parent )
         : TQFrame( parent )
   {
      slider = new TQSlider(TQt::Vertical, this, "volume");
      label = new TQLabel(this);

      mute = new TQToolButton(this, "volume_slider_mute");
      mute->setAutoRaise(true);
      mute->setToggleButton(true);
      TQToolTip::add(mute, i18n("Toggle Mute"));

      TQBoxLayout *lay = new TQVBoxLayout(this);
      lay->addWidget(slider, 0, TQt::AlignHCenter);
      lay->addWidget(label, 0, TQt::AlignHCenter);
      lay->addWidget(mute, 0, TQt::AlignHCenter);
      lay->setMargin(4);

      slider->setRange( 0, 100 );

      setFrameStyle( TQFrame::Plain | TQFrame::Box );
      setSizePolicy( TQSizePolicy::Preferred, TQSizePolicy::Preferred );

      // Test for icon support
      const char* mutedIcons[] = { "audio-volume-muted", "player_mute", "mute" };
      bool iconFound = false;

      for (size_t i = 0; i < (sizeof(mutedIcons) / sizeof(*mutedIcons)); ++i)
      {
         if (!TDEGlobal::iconLoader()->iconPath(mutedIcons[i], TDEIcon::Toolbar, true).isNull())
         {
            mute->setIconSet(TDEGlobal::iconLoader()->loadIconSet(mutedIcons[i], TDEIcon::Toolbar));
            iconFound = true;
            break;
         }
      }

      if (!iconFound)
      {
         mute->setAutoRaise(false);
         mute->setText(i18n("Mute"));
      }

      // Calculate width required for max label size
      label->setText( "100%" );
      adjustSize();
      requiredWidth = width();

      hide();
   }

   TQToolButton *mute;
   TQLabel *label;
   TQSlider *slider;
   int requiredWidth;

   void setMuted(bool muted)
   {
      // Behave correctly when VolumeAction's "setMuted" slot is invoked.
      mute->setDown(muted);
   }
};


VolumeAction::VolumeAction( TDEToolBar *bar, TDEActionCollection *ac )
      : TDEToggleAction( i18n("Volume"), "volume", TQt::Key_1, nullptr, nullptr, ac, "volume" )
      , m_anchor( nullptr )
{
   m_widget = new VolumeSlider( bar->topLevelWidget() );

   connect(this, TQ_SIGNAL(toggled(bool)), TQ_SLOT(toggled(bool)));
   connect(m_widget->mute, TQ_SIGNAL(toggled(bool)), Codeine::engine(), TQ_SLOT(setMuted(bool)));
   connect(m_widget->mute, TQ_SIGNAL(toggled(bool)), TQ_SLOT(setMuted(bool)));
   connect(m_widget->slider, TQ_SIGNAL(valueChanged(int)), Codeine::engine(), TQ_SLOT(setStreamParameter(int)));
   connect(m_widget->slider, TQ_SIGNAL(valueChanged(int)), TQ_SLOT(sliderMoved(int)));
}

int
VolumeAction::plug( TQWidget *bar, int index )
{
   DEBUG_BLOCK

   int const id = TDEAction::plug( bar, index );

   m_anchor = (TQWidget*)bar->child( "toolbutton_volume" ); //TDEAction creates it with this name
   m_anchor->installEventFilter( this ); //so we can keep m_widget anchored

   return id;
}

void
VolumeAction::toggled( bool const b )
{
   m_widget->raise();
   m_widget->setShown( b );
}

void
VolumeAction::sliderMoved(int v)
{
   // TQt sliders are wrong way round when vertical
   v = 100 - v;

   auto vol = TQString("%1%").arg(v);
   m_widget->label->setText(vol);
   setToolTip(i18n("Volume %1").arg(vol));
}

void
VolumeAction::setMuted(bool mute)
{
   m_widget->setMuted(mute);
}

void
VolumeAction::setVolume(int volume)
{
   // TQt sliders are the wrong way round when vertical.
   m_widget->slider->setValue(100 - volume);
}

bool
VolumeAction::eventFilter( TQObject *o, TQEvent *e )
{
   switch (e->type()) {
      case TQEvent::Move:
      case TQEvent::Resize: {
         TQWidget const * const &a = m_anchor;

         m_widget->resize( m_widget->requiredWidth, m_widget->sizeHint().height() );
         m_widget->move( a->mapTo( m_widget->parentWidget(), TQPoint( a->width() - m_widget->width(), a->height() ) ) );
         return false;
      }

      //TODO one click method, flawed currently in fullscreen mode by palette change in mainwindow.cpp
/*      case TQEvent::MouseButtonPress:
         m_widget->show();
         break;

      case TQEvent::MouseButtonRelease:
         m_widget->hide();
         break;*/

      default:
         return false;
   }
}

#include "volumeAction.moc"
